//! `POST /_matrix/identity/*/3pid/unbind`
//!
//! Remove an association between a session and a Matrix user ID.

pub mod v2 {
    //! `/v2/` ([spec])
    //!
    //! [spec]: https://spec.matrix.org/latest/identity-service-api/#post_matrixidentityv23pidunbind

    use ruma_common::{
        OwnedClientSecret, OwnedSessionId, OwnedUserId,
        api::{auth_scheme::AccessToken, request, response},
        metadata,
        thirdparty::Medium,
    };
    use serde::{Deserialize, Serialize};

    metadata! {
        method: POST,
        rate_limited: false,
        authentication: AccessToken,
        history: {
            1.0 => "/_matrix/identity/v2/3pid/unbind",
        }
    }

    /// Request type for the `unbind_3pid` endpoint.
    #[request]
    pub struct Request {
        /// The proof that the client owns the 3PID.
        ///
        /// If this is not provided, the request must be signed by the homeserver which controls
        /// the `mxid`.
        #[serde(flatten, skip_serializing_if = "Option::is_none")]
        pub threepid_ownership_proof: Option<ThreePidOwnershipProof>,

        /// The Matrix user ID to remove from the 3PIDs.
        pub mxid: OwnedUserId,

        /// The 3PID to remove.
        ///
        /// Must match the 3PID used to generate the session if using `sid` and `client_secret` to
        /// authenticate this request.
        pub threepid: ThirdPartyId,
    }

    /// Response type for the `unbind_3pid` endpoint.
    #[response]
    #[derive(Default)]
    pub struct Response {}

    impl Request {
        /// Creates a `Request` with the given Session ID, client secret, Matrix user ID and 3PID.
        pub fn new(
            threepid_ownership_proof: Option<ThreePidOwnershipProof>,
            mxid: OwnedUserId,
            threepid: ThirdPartyId,
        ) -> Self {
            Self { threepid_ownership_proof, mxid, threepid }
        }
    }

    impl Response {
        /// Creates an empty `Response`.
        pub fn new() -> Self {
            Self {}
        }
    }

    /// A 3PID to unbind.
    #[derive(Clone, Debug, Serialize, Deserialize)]
    #[cfg_attr(not(ruma_unstable_exhaustive_types), non_exhaustive)]
    pub struct ThirdPartyId {
        /// A medium matching the medium of identifier to unbind.
        pub medium: Medium,

        /// The 3PID address to remove.
        pub address: String,
    }

    impl ThirdPartyId {
        /// Creates a new `ThirdPartyId` with the given medium and address.
        pub fn new(medium: Medium, address: String) -> Self {
            Self { medium, address }
        }
    }

    /// A proof that the client owns the 3PID.
    ///
    /// Must be constructed using the same session ID and client secret generated and passed by the
    /// `requestToken` call for the given 3PID.
    #[derive(Clone, Debug, Serialize, Deserialize)]
    #[cfg_attr(not(ruma_unstable_exhaustive_types), non_exhaustive)]
    pub struct ThreePidOwnershipProof {
        /// The Session ID generated by the `requestToken` call.
        pub sid: OwnedSessionId,

        /// The client secret passed to the `requestToken` call.
        pub client_secret: OwnedClientSecret,
    }

    impl ThreePidOwnershipProof {
        /// Creates a new `ThreePidOwnershipProof` with the given session ID and client secret.
        pub fn new(sid: OwnedSessionId, client_secret: OwnedClientSecret) -> Self {
            Self { sid, client_secret }
        }
    }
}
