/**
 * \file sage2psmcborcli.cpp
 * \date 22/01/2026
 * \author Olivier Langella
 * \brief CLI tool to convert sage tsv results to psmCBOR
 */


/*******************************************************************************
 * Copyright (c) 2026 Olivier Langella
 *<Olivier.Langella@universite-paris-saclay.fr>.
 *
 * This file is part of the PAPPSOms++ library.
 *
 *     PAPPSOms++ is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     PAPPSOms++ is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with PAPPSOms++.  If not, see <http://www.gnu.org/licenses/>.
 *
 ******************************************************************************/


#include "sage2psmcborcli.h"
#include "pappsomspp/config.h"
#include <QIODevice>
#include <QCommandLineParser>
#include "pappsomspp/core/processing/uimonitor/uimonitortext.h"
#include "pappsomspp/core/processing/cbor/psm/sage/sagefilereader.h"
#include "pappsomspp/core/pappsoexception.h"
#include <QDateTime>
#include <QDir>
#include <QFile>
#include <QFileInfo>
#include <QTimer>
#include <sys/stat.h>

Sage2PsmcborCli::Sage2PsmcborCli(QObject *parent) : QObject(parent)
{
  // get the instance of the main application
  app = QCoreApplication::instance();
  // setup everything here
  // create any global objects
  // setup debug and warning mode
}

Sage2PsmcborCli::~Sage2PsmcborCli()
{
}

// 10ms after the application starts this method will run
// all QT messaging is running at this point so threads, signals and slots
// will all work as expected.
void
Sage2PsmcborCli::run()
{
  qSetMessagePattern(QString("%{file}@%{line}, %{function}(): %{message}"));

  QTextStream errorStream(stderr, QIODevice::WriteOnly);
  QTextStream outputStream(stderr, QIODevice::WriteOnly);

  try
    {
      qDebug();
      QCommandLineParser parser;

      // throw pappso::PappsoException("test");
      // throw pappso::PappsoException("test");
      parser.setApplicationDescription(QString("sage2psmcbor")
                                         .append(" ")
                                         .append(PAPPSOMSPP_VERSION)
                                         .append(" Sage to psmCBOR converter"));

      parser.addHelpOption();
      parser.addVersionOption();
      parser.addPositionalArgument("JSON", tr("Sage JSON result file to read"));

      QCommandLineOption psmcborOutputOption(
        QStringList() << "o" << "output",
        QCoreApplication::translate("psmCBOR file", "path to the psmCBOR output file <psmcbor>."),
        QCoreApplication::translate("psmcbor", "psmcbor"));

      parser.addOption(psmcborOutputOption);

      qDebug();

      // Process the actual command line arguments given by the user
      parser.process(*app);

      // QCoreApplication * app(this);
      // Add your main code here
      // qDebug();
      const QDateTime dt_begin = QDateTime::currentDateTime();
      const QStringList args   = parser.positionalArguments();


      qDebug();
      pappso::UiMonitorText monitor(outputStream);


      // PsmProcess cbor_psm_process(json_parameters);
      QFileInfo input_sage_file;
      if(parser.positionalArguments().size() == 1)
        {
          input_sage_file.setFile(parser.positionalArguments().front());
        }
      else
        {
          throw pappso::PappsoException(
            QObject::tr("input Sage file is not set. Please provide the path to the sage JSON "
                        "result file in positional arguments"));
        }

      QFile mzcbor_file;
      pappso::cbor::CborStreamWriter *p_cborWriter = nullptr;
      QString mzcbor_index_filename;
      if(parser.isSet(psmcborOutputOption))
        {
          QString mzcbor_filename = parser.value(psmcborOutputOption);
          if(!mzcbor_filename.isEmpty())
            {
              mzcbor_file.setFileName(mzcbor_filename);
              mzcbor_file.open(QIODevice::WriteOnly);
              p_cborWriter          = new pappso::cbor::CborStreamWriter(&mzcbor_file);
              mzcbor_index_filename = mzcbor_filename.append(".idx");
            }
        }
      else
        {
          mzcbor_file.open(stdout, QIODevice::WriteOnly);
          p_cborWriter = new pappso::cbor::CborStreamWriter(&mzcbor_file);
        }


      if(p_cborWriter != nullptr)
        {
          pappso::cbor::psm::SageFileReader sage_reader(&monitor, p_cborWriter, input_sage_file);

          mzcbor_file.close();

          delete p_cborWriter;
        }


      qDebug();
    }
  catch(pappso::PappsoException &error)
    {

      errorStream << QString("Oops! an error occurred in %1. Don't Panic :\n%2")
                       .arg(QCoreApplication::applicationName())
                       .arg(error.qwhat());

      errorStream << Qt::endl << Qt::endl;

      errorStream.flush();
      app->exit(1);
      exit(1);
    }

  catch(std::exception &error)
    {

      errorStream << QString("Oops! an error occurred in %1. Don't Panic :\n%2")
                       .arg(QCoreApplication::applicationName())
                       .arg(error.what());

      errorStream << Qt::endl << Qt::endl;

      errorStream.flush();
      app->exit(1);
      exit(1);
    }


  // you must call quit when complete or the program will stay in the
  // messaging loop
  quit();
}

// call this routine to quit the application
void
Sage2PsmcborCli::quit()
{
  // you can do some cleanup here
  // then do emit finished to signal CoreApplication to quit
  emit finished();
}

// shortly after quit is called the CoreApplication will signal this routine
// this is a good place to delete any objects that were created in the
// constructor and/or to stop any threads
void
Sage2PsmcborCli::aboutToQuitApp()
{
  // stop threads
  // sleep(1);   // wait for threads to stop.
  // delete any objects
}

int
main(int argc, char **argv)
{
  umask(0);
  // QTextStream consoleErr(stderr);
  // QTextStream consoleOut(stdout, QIODevice::WriteOnly);
  // ConsoleOut::setCout(new QTextStream(stdout, QIODevice::WriteOnly));
  // ConsoleOut::setCerr(new QTextStream(stderr, QIODevice::WriteOnly));
  qDebug();
  QCoreApplication app(argc, argv);
  qDebug();
  QCoreApplication::setApplicationName("sage2psmcbor");
  QCoreApplication::setApplicationVersion(PAPPSOMSPP_VERSION);
  QLocale::setDefault(QLocale::system());

  // create the main class
  Sage2PsmcborCli myMain;
  // connect up the signals
  QObject::connect(&myMain, SIGNAL(finished()), &app, SLOT(quit()));
  QObject::connect(&app, SIGNAL(aboutToQuit()), &myMain, SLOT(aboutToQuitApp()));
  qDebug();


  // This code will start the messaging engine in QT and in
  // 10ms it will start the execution in the MainClass.run routine;
  QTimer::singleShot(10, &myMain, SLOT(run()));
  return app.exec();
}
