/*************************************************************************
* Copyright (C) 2013 Intel Corporation
*
* Licensed under the Apache License,  Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
* 	http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law  or agreed  to  in  writing,  software
* distributed under  the License  is  distributed  on  an  "AS IS"  BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the  specific  language  governing  permissions  and
* limitations under the License.
*************************************************************************/

/*
//
//  Purpose:
//     Cryptography Primitive.
//     AES encryption/decryption (CBC mode)
//     AES encryption/decryption (CBC-CS mode)
//
//  Contents:
//        ippsAESEncryptCBC()
//
*/

#include "owndefs.h"
#include "owncp.h"
#include "pcpaesm.h"
#include "pcptool.h"
#include "pcpaes_cbc_encrypt.h"

#if (_ALG_AES_SAFE_ == _ALG_AES_SAFE_COMPACT_SBOX_)
#include "pcprijtables.h"
#endif

/*F*
//    Name: ippsAESEncryptCBC
//
// Purpose: AES-CBC encryption.
//
// Returns:                Reason:
//    ippStsNullPtrErr        pCtx == NULL
//                            pSrc == NULL
//                            pDst == NULL
//                            pIV  == NULL
//    ippStsContextMatchErr   !VALID_AES_ID()
//    ippStsLengthErr         len <1
//    ippStsUnderRunErr       0!=(dataLen%MBS_RIJ128)
//    ippStsNoErr             no errors
//
// Parameters:
//    pSrc        pointer to the source data buffer
//    pDst        pointer to the target data buffer
//    len         input/output buffer length (in bytes)
//    pCtx        pointer to the AES context
//    pIV         pointer to the initialization vector
//
*F*/

/* clang-format off */
IPPFUN(IppStatus, ippsAESEncryptCBC, (const Ipp8u* pSrc,
                                      Ipp8u* pDst,
                                      int len,
                                      const IppsAESSpec* pCtx,
                                      const Ipp8u* pIV))
/* clang-format on */
{
    /* test context */
    IPP_BAD_PTR1_RET(pCtx);
    /* test the context ID */
    IPP_BADARG_RET(!VALID_AES_ID(pCtx), ippStsContextMatchErr);

    /* test source, target buffers and initialization pointers */
    IPP_BAD_PTR3_RET(pSrc, pIV, pDst);
    /* test stream length */
    IPP_BADARG_RET((len < 1), ippStsLengthErr);
    /* test stream integrity */
    IPP_BADARG_RET((len & (MBS_RIJ128 - 1)), ippStsUnderRunErr);

    /* do encryption */
    cpEncryptAES_cbc(pIV, pSrc, pDst, len / MBS_RIJ128, pCtx);

    return ippStsNoErr;
}
